/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.client.api.Connection;
import org.eclipse.jetty.client.api.Destination;
import org.eclipse.jetty.util.BlockingArrayQueue;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.thread.SpinLock;

public class ConnectionPool
implements Closeable,
Dumpable {
    protected static final Logger LOG = Log.getLogger(ConnectionPool.class);
    private final AtomicInteger connectionCount = new AtomicInteger();
    private final SpinLock lock = new SpinLock();
    private final Destination destination;
    private final int maxConnections;
    private final Promise<Connection> requester;
    private final BlockingDeque<Connection> idleConnections;
    private final BlockingQueue<Connection> activeConnections;

    public ConnectionPool(Destination destination, int maxConnections, Promise<Connection> requester) {
        this.destination = destination;
        this.maxConnections = maxConnections;
        this.requester = requester;
        this.idleConnections = new LinkedBlockingDeque<Connection>(maxConnections);
        this.activeConnections = new BlockingArrayQueue<Connection>(maxConnections);
    }

    public int getConnectionCount() {
        return this.connectionCount.get();
    }

    public BlockingQueue<Connection> getIdleConnections() {
        return this.idleConnections;
    }

    public BlockingQueue<Connection> getActiveConnections() {
        return this.activeConnections;
    }

    public Connection acquire() {
        Connection connection = this.activateIdle();
        if (connection == null) {
            connection = this.tryCreate();
        }
        return connection;
    }

    private Connection tryCreate() {
        int next;
        int current;
        do {
            if ((next = (current = this.getConnectionCount()) + 1) <= this.maxConnections) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Max connections {}/{} reached", current, this.maxConnections);
            }
            return this.activateIdle();
        } while (!this.connectionCount.compareAndSet(current, next));
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connection {}/{} creation", next, this.maxConnections);
        }
        this.destination.newConnection(new Promise<Connection>(){

            @Override
            public void succeeded(Connection connection) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Connection {}/{} creation succeeded {}", next, ConnectionPool.this.maxConnections, connection);
                }
                ConnectionPool.this.idleCreated(connection);
                ConnectionPool.this.requester.succeeded(connection);
            }

            @Override
            public void failed(Throwable x) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Connection " + next + "/" + ConnectionPool.this.maxConnections + " creation failed", x);
                }
                ConnectionPool.this.connectionCount.decrementAndGet();
                ConnectionPool.this.requester.failed(x);
            }
        });
        return this.activateIdle();
    }

    protected void idleCreated(Connection connection) {
        boolean idle;
        try (SpinLock.Lock lock = this.lock.lock();){
            idle = this.idleConnections.offerLast(connection);
        }
        this.idle(connection, idle);
    }

    private Connection activateIdle() {
        boolean acquired;
        Connection connection;
        try (SpinLock.Lock lock = this.lock.lock();){
            connection = (Connection)this.idleConnections.pollFirst();
            if (connection == null) {
                Connection connection2 = null;
                return connection2;
            }
            acquired = this.activeConnections.offer(connection);
        }
        if (acquired) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection active {}", connection);
            }
            this.acquired(connection);
            return connection;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connection active overflow {}", connection);
        }
        connection.close();
        return null;
    }

    protected void acquired(Connection connection) {
    }

    public boolean release(Connection connection) {
        boolean idle;
        try (SpinLock.Lock lock = this.lock.lock();){
            if (!this.activeConnections.remove(connection)) {
                boolean bl = false;
                return bl;
            }
            idle = this.idleConnections.offerFirst(connection);
        }
        this.released(connection);
        return this.idle(connection, idle);
    }

    protected boolean idle(Connection connection, boolean idle) {
        if (idle) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection idle {}", connection);
            }
            return true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connection idle overflow {}", connection);
        }
        connection.close();
        return false;
    }

    protected void released(Connection connection) {
    }

    public boolean remove(Connection connection) {
        boolean removed;
        boolean idleRemoved;
        boolean activeRemoved;
        try (SpinLock.Lock lock = this.lock.lock();){
            activeRemoved = this.activeConnections.remove(connection);
            idleRemoved = this.idleConnections.remove(connection);
        }
        if (activeRemoved) {
            this.released(connection);
        }
        boolean bl = removed = activeRemoved || idleRemoved;
        if (removed) {
            int pooled = this.connectionCount.decrementAndGet();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection removed {} - pooled: {}", connection, pooled);
            }
        }
        return removed;
    }

    public boolean isActive(Connection connection) {
        try (SpinLock.Lock lock = this.lock.lock();){
            boolean bl = this.activeConnections.contains(connection);
            return bl;
        }
    }

    public boolean isIdle(Connection connection) {
        try (SpinLock.Lock lock = this.lock.lock();){
            boolean bl = this.idleConnections.contains(connection);
            return bl;
        }
    }

    public boolean isEmpty() {
        return this.connectionCount.get() == 0;
    }

    @Override
    public void close() {
        ArrayList<Connection> idles = new ArrayList<Connection>();
        ArrayList<Connection> actives = new ArrayList<Connection>();
        try (SpinLock.Lock lock = this.lock.lock();){
            idles.addAll(this.idleConnections);
            this.idleConnections.clear();
            actives.addAll(this.activeConnections);
            this.activeConnections.clear();
        }
        this.connectionCount.set(0);
        for (Connection connection : idles) {
            connection.close();
        }
        for (Connection connection : actives) {
            connection.close();
        }
    }

    @Override
    public String dump() {
        return ContainerLifeCycle.dump(this);
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        ArrayList<Connection> actives = new ArrayList<Connection>();
        ArrayList<Connection> idles = new ArrayList<Connection>();
        try (SpinLock.Lock lock = this.lock.lock();){
            actives.addAll(this.activeConnections);
            idles.addAll(this.idleConnections);
        }
        ContainerLifeCycle.dumpObject(out, this);
        ContainerLifeCycle.dump(out, indent, actives, idles);
    }

    public String toString() {
        int idleSize;
        int activeSize;
        try (SpinLock.Lock lock = this.lock.lock();){
            activeSize = this.activeConnections.size();
            idleSize = this.idleConnections.size();
        }
        return String.format("%s[c=%d/%d,a=%d,i=%d]", this.getClass().getSimpleName(), this.connectionCount.get(), this.maxConnections, activeSize, idleSize);
    }
}

